% fault_slip_code_comparison
%
% Computes and generates plots for code comparison report.
% Can only be run stand-alone. 
%

clear variables
close all

% ---------------------------------------------------------------------------------------------
% Set local plot flags:
% ---------------------------------------------------------------------------------------------
flag_plot_cap = 0; % flag for plot captions (black) and plot file location captions (red)
flag_plot_no_offset = 1;
flag_plot_no_friction = 1;
flag_plot_constant_friction = 1;
flag_plot_slip_weakening_friction = 0;
flag_plot_offset_sensitivity = 0; % warning: takes quite some time
flag_plot_some_more = 0;
% 0 = off
% 1 = on

% ---------------------------------------------------------------------------------------------
% Set generic input parameters (plot-specific ones will be set later):
% ---------------------------------------------------------------------------------------------
% Parameter set for Example 1 from paper:
a = 75; % see Figure 1, m
alpha = 0.9; % Biot's coefficient, -
b = 150; % see Figure 1, m
delta_c = 0.02; % critical slip distance for slip weakening, m
D_center = 3500; % reservoir depth @ y = 0, m
g = 9.81; % acceleration of gravity, m/s^2
G = 6500e6; % shear modulus, Pa
kappa = 0; % cohesion, -
K_eff = 0.5; % initial effective vertical/horizontal stress ratio, -
mu_dyn = 0.20; % dynamic friction coefficient, -
mu_st = 0.52; % static friction coefficient, -
nu = 0.15; % Poisson's ratio, -
p_0_center = 35e6; % initial pressure @ y = 0, Pa
phi = 0.15; % porosity, -
rho_f = 1020; % fluid density, kg/m^3
rho_s = 2650; % solid density, kg/m^3
sigma_v_0 = 0; % initial vertical stress @ y = 0, Pa. If set equal to zero, the stress is ...
% computed from the weigth of the overburden; see function fault_slip_sigma.m.
theta = 70*pi/180; % dip angle, rad

% ---------------------------------------------------------------------------------------------
% Set generic program flow flags and computational parameters:
% ---------------------------------------------------------------------------------------------
% Program flow flags:
f_damp = 0.5; % damping factor for iterative update of slip-weakening and coupled ...
    % simulation (only relevant for flag_cheb == 1)
flag_cheb = 0; % flag to trigger use of Chebyshev polynomials for numerical integration
% 0 = numerical integration with near-uniform staggered grids for y and xi
% 1 = semi-analytical integration with Chebyshev polynomials
flag_check = 0; % flag to trigger graphical check of finite difference and Chebyshev results
    % 0 = check disabled (default) 
    % 1 = check enabled (for diagnostics; causes overwriting of all input from main file; ...
    %     see fault_slip_patch_bounds_check.m)
flag_cont = 1; % flag to force continuation in case of non-convergence of slip-weakening ...
    % and coupled simulation (only relevant for flag_cheb == 1)
    % 0 = continuation disabled (for diagnostics) 
    % 1 = continuation enabled (recommended for regular use)
flag_cor = 1; % flag to force correct sign on slip gradient (only relevant for regular integr.)
% 0 = forcing switched off (for diagnostics)
% 1 = forcing switched on (recommended for regular use)
flag_decouple = 0; % flag to force decoupling between slip patches (only relevant for ...
% slip-weakening friction)
% 0 = default coupled simulation; correct but takes more time; not always necessary
% 1 = decoupled simulation; faster but possibly inaccurate (only available for ...
%     slip-weakening or rate and state-dependent friction)
flag_echo = 1; % flag to trigger writing to screen of simulation progress
% 0 = no echo
% 1 = limited echo; only info about main simulation elements
% 2 = full echo; info about each simulation steps
flag_eig_use = 2; % flag to determine use of eigenvalues (only relevant for flag_fric > 1)
    % 0 = used to check accuracy of simulated slip length (Delta y*_sim ~= Delta y*_eig?) 
    %     (for diagnostics)
    % 1 = used for seismicity detection (stops simulation when Delta y*_sim > Delta y*_eig)
    %     (recommended for regular use)
    % 2 = alternative: numerical derivative used for seismicity detection (stops simulation 
    %     when (dp/dDelta y) drops below tol_deriv); requires flag_lag = 0  
    %     (for diagnostics)
tol_deriv = 1e-5; % tolerance for scaled numerical derivative (dp/dDelta y)*(Delta_y/A), ...
    % where A = G / (2*pi*(1-nu)), - 
flag_lag = 0; % flag to trigger lagging of eigen problem computation with one presure step
% 0 = lagging switched off (for diagnostics)
% 1 = lagging switched on (recommended for regular use)
flag_M = 0; % flag to compute seismic moment (can be time consuming)
    % 0 = computation switched off
    % 1 = computation switched on
flag_paper = 1; % flag to trigger plotting of figure for paper
% 0 = default plots
% 1 = plot for paper; causes overwriting of all input from main file; check ...
%     fault_slip_patch_bounds_all.m
flag_reg = 1; % flag to control regularization of slip-dependent friction coefficient and ...
% Coulomb stresses:
% 0 = regularization switched off (for diagnostics)
% 1 = regularization switched on (recommended for regular use)
eps_reg = 0; % regularization parameter for slip-dependent friction coefficient, m
eta_reg = 0.10; % regularization parameter for Coulomb stresses, m
% eta_reg = 0.20; % regularization parameter for Coulomb stresses, m
flag_top_only = 1; % flag to force occurrence of seismicity in top patch only (only ...
    % relevant for flag_cheb == 1)  
    % 0 = forcing switched off (for diagnostics)
    % 1 = forcing switched on (recommended for regular use)

% Computational parameters:
n_max_int = 2000; % max. allowed total nr. of integr. pnts, -. Only relevant for flag_cheb = 0
n_min_int = 100;  % min. required total nr. of integr. pnts, -. Only relevant for flag_cheb = 0
cheb_terms = 250; % Number of terms in Chebyshev polynomial approximation for semi-...
% analytical integration. Only relevant for flag_cheb = 1
max_iter_coupled = 500; % maximum allowed number of iterations
p_max_bounds = -12e6; % highest incremental pressure (negative for production), Pa
p_min_bounds = -30e6; % lowest incremental pressure (negative for production), Pa
Delta_p_init = -0.5e6; % initial step size (negative for production), Pa
Delta_p_max = -1.0e6; % maximum allowed step size (negative for production), Pa
% Delta_p_max = -0.1e6; % maximum allowed step size (negative for production), Pa
small_patch = 0.05; % minimum patch size to trigger slip computation during pr. stepping, m
% small_patch = 0.001; % minimum patch size to trigger slip computation during pr. stepping, m

% ---------------------------------------------------------------------------------------------
% Set generic plotting flags and parameters:
% ---------------------------------------------------------------------------------------------
% Plotting flags:
flag_plot_diag = 0; % flag for diagnostic plots:
% 0 = plotting switched off
% 1 = plotting switched on

% Parameters for snapshot plots of stresses, slip etc.:
% Note 1: all comments refer to parameter settings for example 1 from the paper.
% Note 2: different plots are created for zero, constant or slip-weakening ...
% friction. Moreover for p_inc_02, additional plots are created in case of ...
% slip-weakening friction. See fault_slip_patch_bounds.m for further info.
y_plot_max_delta_1 =  100; % covers both slip patches (in detail), m
y_plot_min_delta_1 = -100; % covers both slip patches (in detail), m
y_plot_max_delta_2 =   80; % covers top patch only (further detail), m
y_plot_min_delta_2 =   60; % covers top patch only (further detail), m
y_plot_max_sigma   =  250; % covers both slip patches (overview), m
y_plot_min_sigma   = -250; % covers both slip patches (overview), m
y_plot_n_pt = 3000; % number of points between y_plot_min_...and y_plot_max_...

% Parameters for plots of Coulomb stress zeros and slip patch boundaries as a function of
% depletion pressure:
y_plot_max_bounds =  100; % covers both slip patches (in detail), m
y_plot_min_bounds = -100; % covers both slip patches (in detail), m

% Colors for line plots:
blue = [0/255 128/255 255/255];
gray = [235/255 235/255 235/255];
green = [0/255 204/255 0/255];
orange = [255/255 185/255 0/255];
colors = [blue; gray; green; orange];

% ---------------------------------------------------------------------------------------------
% End of generic input. Start of plot-specific input.
% ---------------------------------------------------------------------------------------------

%%%%%%%%%%%%%%%%%%%%
% Part 0: No offset:
%%%%%%%%%%%%%%%%%%%%
if flag_plot_no_offset == 1
    % Initial stresses:
    % Compute analytical data:
    theta_deg = 70;
    theta = theta_deg*pi/180;
    y = linspace(-2250,2250,500);
    sigma_xx_ini = -57.05e6 + 16.30e3 * y;
    sigma_yy_ini = -82.60e6 + 23.60e3 * y;
    sigma_norm_ini = sigma_xx_ini * sin(theta)^2 + sigma_yy_ini * cos(theta)^2;
    sigma_shear_ini = (sigma_xx_ini - sigma_yy_ini) * sin(theta)*cos(theta);
    % sigma_norm_ini_2 = -60.04e6 + 17.15e3 * y; % check
    % sigma_shear_ini_2 = 8.21e6 - 2.35e3 * y; % check
    
    % Read DARTS data:
    data_DARTS_initial_fault_stresses = readmatrix('data_DARTS_initial_fault_stresses.txt');
    y_DARTS              = data_DARTS_initial_fault_stresses(:,1);
    sigma_norm_ini_DARTS = data_DARTS_initial_fault_stresses(:,2);
    sigma_shear_ini_DARTS = data_DARTS_initial_fault_stresses(:,3);
    
    % Create plot:
    subplot(1,2,1)
    plot(sigma_norm_ini/1e6,y,'-o','Color','r','LineWidth',1.0,'MarkerIndices',1:50:length(y),...
        'MarkerSize',5.0)
    hold on
    plot(sigma_norm_ini_DARTS/10,y_DARTS,'-','Color',blue,'LineWidth',1.0)
    hold off
    xlabel('$\it \sigma_{\perp}^0$ (MPa)','Interpreter','latex','FontSize',11)
    ylabel('$\it y$ (m)','Interpreter','latex','FontSize',11)
    axis([-inf inf -2250 2250])
    legend('Analytical','DARTS','location','northwest')
    
    subplot(1,2,2)
    plot(sigma_shear_ini/1e6,y,'-o','Color','r','LineWidth',1.0,'MarkerIndices',1:50:length(y),...
        'MarkerSize',5.0)
    hold on
    plot(sigma_shear_ini_DARTS/10,y_DARTS,'-','Color',blue,'LineWidth',1.0)
    hold off
    xlabel('$\it \sigma_{\parallel}^0$ (MPa)','Interpreter','latex','FontSize',11)
    % ylabel('$\it y$ (m)','Interpreter','latex','FontSize',11)
    axis([-inf inf -2250 2250])
    
    % Combined (initial plus incremental) stresses:
    % Set specific variables:
    a = 112.5;
    b = 112.5;
    flag_fric = 1;
    p_inc = -25.0e6;
    
    % Compute auxiliary variables:
    A = G / (2*pi*(1-nu));
    C = ((1-2*nu)*alpha*p_inc) / (2*pi*(1-nu)); % scaling parameter, N/m^2
    
    % Compute stresses at the fault excluding the effect of fault slip:
    [p_0,p_comb,p_comb_full,...
        sigma_C,sigma_C_full,...
        sigma_norm_0,sigma_norm,sigma_norm_comb,...
        sigma_norm_eff_0,sigma_norm_eff,sigma_norm_eff_comb,...
        sigma_norm_eff_full,sigma_norm_eff_full_comb,...
        sigma_shear_0,sigma_shear,sigma_shear_comb,...
        sigma_slip,sigma_slip_full,y] = ...
        fault_slip_sigma(a,alpha,b,0,0,D_center,eps_reg,eta_reg,0,flag_reg,g,kappa,K_eff,0,...
        mu_st,y_plot_n_pt,nu,p_0_center,p_inc,phi,rho_f,rho_s,sigma_v_0,theta,0,y_plot_max_sigma,...
        y_plot_min_sigma);
    
    % Read DARTS data:
    data_DARTS_incremental_fault_stresses = readmatrix('data_DARTS_incremental_fault_stresses.txt');
    y_DARTS              = data_DARTS_incremental_fault_stresses(:,1);
    sigma_norm_inc_DARTS = data_DARTS_incremental_fault_stresses(:,2);
    sigma_shear_inc_DARTS = data_DARTS_incremental_fault_stresses(:,3);
    
    % Plot stresses:
    figure
    subplot(1,2,1)
    hold on
    x_min = -70;
    x_max = -40;
    y_min = y_plot_min_sigma;
    y_max = y_plot_max_sigma;
    line([0 0],[y_min y_max],'LineStyle','-','Color','k','LineWidth',0.5)
    line([x_min x_max],[112.5 112.5],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[-112.5 -112.5],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[0 0],'LineStyle','-','Color','k','LineWidth',0.5);
    p1 = plot(sigma_norm_comb/1e6,y,'-o','Color','r','LineWidth',1.0,'MarkerIndices',...
        1:200:length(y),'MarkerSize',5.0);
    p2 = plot(sigma_norm_inc_DARTS/10,y_DARTS,'-','Color',blue,'LineWidth',1.0);
    hold off
    xlabel('$\Sigma_{\perp} \rm(MPa)$','Interpreter','latex','FontSize',11)
    ylabel('$\it y \rm(m)$','Interpreter','latex','FontSize',11)
    legend([p1 p2],'Analytical','DARTS','location','southeast')
    axis([x_min x_max y_min y_max])
    box on
    
    subplot(1,2,2)
    hold on
    x_min = 5;
    x_max = 15;
    y_min = y_plot_min_sigma;
    y_max = y_plot_max_sigma;
    line([0 0],[y_min y_max],'LineStyle','-','Color','k','LineWidth',0.5)
    line([x_min x_max],[112.5 112.5],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[-112.5 -112.5],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[0 0],'LineStyle','-','Color','k','LineWidth',0.5)
    plot(sigma_shear_comb/1e6,y,'-o','Color','r','LineWidth',1.0,'MarkerIndices',...
        1:200:length(y),'MarkerSize',5.0)
    plot(sigma_shear_inc_DARTS/10,y_DARTS,'-','Color',blue,'LineWidth',1.0);
    hold off
    xlabel('$\Sigma_{\parallel} \rm(MPa)$','Interpreter','latex','FontSize',11)
    %    ylabel('$\it y \rm(m)$','Interpreter','latex','FontSize',11)
    axis([x_min x_max y_min y_max])
    box on
    
    % Reset variables:
    a = 75;
    b = 150;
end

%%%%%%%%%%%%%%%%%%%%%%%
% Part 1: No friction:
%%%%%%%%%%%%%%%%%%%%%%%
if flag_plot_no_friction == 1
    % Set specific variables:
    flag_code_comp = 1; % local flag
    p_inc = -25e6;
    theta = pi/2; % vertical fault
    mu_st = 0; % frictionless fault
    
    % Compute auxiliary variables:
    A = G / (2*pi*(1-nu));
    C = ((1-2*nu)*alpha*p_inc) / (2*pi*(1-nu)); % scaling parameter, N/m^2
    
    % Compute stresses at the fault excluding the effect of fault slip:
    [~,~,~,sigma_C,~,~,~,~,~,~,~,~,~,~,~,~,~,~,y_sigma] = ...
        fault_slip_sigma(a,alpha,b,0,0,D_center,eps_reg,eta_reg,0,flag_reg,g,kappa,K_eff,0,...
        mu_st,y_plot_n_pt,nu,p_0_center,p_inc,phi,rho_f,rho_s,sigma_v_0,theta,0,y_plot_max_sigma,...
        y_plot_min_sigma);
    
%     % Check stresses:
%     y_check = linspace(-2250,2250,500);
%     sigma_C_check = C/2 * log( (y_check-75).^2.*(y_check+75).^2./...
%         ((y_check-150).^2.*(y_check+150).^2));
    
    % Compute slip:
    y_delta = linspace(y_plot_min_sigma,y_plot_max_sigma,y_plot_n_pt);
    delta = ...
        0 * y_delta          .* ( y_delta <= -b       ) ...
        + (C/A) * (-y_delta-b) .* (-b        <   y_delta) .* (y_delta < -a) ...
        + (C/A) * (a-b)        .* (-a        <   y_delta) .* (y_delta <  a) ...
        + (C/A) * (y_delta-b)  .* ( a        <   y_delta) .* (y_delta <  b) ...
        + 0 * y_delta          .* ( b        <=  y_delta);
    
    % Read DARTS data:
    data_DARTS_vertical_line = readmatrix('data_DARTS_vertical_line.txt');
    y_DARTS_1 = data_DARTS_vertical_line(:,1);
    Sigma_C_DARTS = data_DARTS_vertical_line(:,3);
    data_DARTS_vertical_frictionless_fault = readmatrix('data_DARTS_vertical_frictionless_fault.txt');
    y_DARTS_2 = data_DARTS_vertical_frictionless_fault(:,1);
    delta_DARTS = data_DARTS_vertical_frictionless_fault(:,4);
    
    % Plot Coulomb stress:
    figure
    subplot(1,2,1)
    hold on
    x_min = -20;
    x_max = 20;
    y_min = y_plot_min_sigma;
    y_max = y_plot_max_sigma;
    line([0 0],[y_min y_max],'LineStyle','-','Color','k','LineWidth',0.5)
    line([x_min x_max],[150 150],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[75 75],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[-75 -75],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[-150 -150],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[0 0],'LineStyle','-','Color','k','LineWidth',0.5)
    plot(sigma_C/1e6,y_sigma,'-o','Color','r','LineWidth',1.0,'MarkerIndices',...
        1:200:length(y_sigma),'MarkerSize',5.0)
    plot(Sigma_C_DARTS/10,y_DARTS_1,'-','Color',blue,'LineWidth',1.0)
    hold off
    xlabel('$\Sigma_C \,(= \sigma_{xy})$ (MPa) ','Interpreter','latex','FontSize',11)
    ylabel('$\it y \rm(m)$','Interpreter','latex','FontSize',11)
%     axis([x_min x_max y_min y_max])
    box on
    
    subplot(1,2,2)
    hold on
    x_min = 0;
    x_max = 0.20;
    y_min = y_plot_min_sigma;
    y_max = y_plot_max_sigma;
    line([x_min x_max],[150 150],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[75 75],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[-75 -75],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[-150 -150],'LineStyle',':','Color','k','LineWidth',1)
    line([x_min x_max],[0 0],'LineStyle','-','Color','k','LineWidth',0.5)
    p1 = plot(delta,y_delta,'-o','Color','r','LineWidth',1.0,'MarkerIndices',...
        1:200:length(y_delta),'MarkerSize',5.0);
    p2 = plot(delta_DARTS,y_DARTS_2,'-','Color',blue,'LineWidth',1.0);
    hold off
    xlabel('$\delta$ (m) ','Interpreter','latex','FontSize',11)
%    ylabel('$\it y \rm(m)$','Interpreter','latex','FontSize',11)
    legend([p1 p2],'Analytical','DARTS','location','southeast')
    axis([x_min x_max y_min y_max])
    box on
    
    % Reset variables:
    mu_st = 0.52;
    theta = 70*pi/180;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Part 2: Constant friction:
%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if flag_plot_constant_friction == 1
    flag_fric = 1;
    Delta_y_int = 0.10; % integration step size, m
    tol_rel_coupled = 1e-4; % relative tolerance for convergence of coupling and slip weakening
    Delta_p_min = -0.10e6; % minimum allowed step size (negative for production), Pa
    Delta_y_target = 0.50; % target change in y value, m
    p_inc_01 = -24e6; % incremental pressure at which to take a snapshot, Pa
    p_inc_02 = -25e6;
    p_inc_03 = -26e6;
    p_inc_04 = -27e6;
    p_inc_05 = -28e6;
    flag_plot_eig = 0; % flag for eigenvalue plot; plotting switched off
    flag_plot_snap = 1; % flag for snapshot plots; plotting switched on
    
    [~,~,~,~,~,p_plot,p_2_plot,p_3_plot,~,p_tilde_2_plot,p_tilde_3_plot,...
        p_tilde_unc_2_plot,p_tilde_unc_3_plot,y_1_plot,y_2_plot,y_3_plot,y_4_plot,...
        y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,...
        y_tilde_unc_2_plot,y_tilde_unc_3_plot] = ...
        fault_slip_patch_bounds_code_comparison(a,alpha,b,cheb_terms,colors,delta_c,...
        D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
        eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
        flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
        flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
        g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
        p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
        p_max_bounds,p_min_bounds,...
        phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
        y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
        y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
        y_plot_n_pt);
    
    % Create first part of plot:
    % Read DARTS data:
    data_DARTS_constant_friction_patch_boundaries = ...
        readmatrix('data_DARTS_constant_friction_patch_boundaries.txt');
    p_DARTS = data_DARTS_constant_friction_patch_boundaries(:,1);
    y_tilde_DARTS = data_DARTS_constant_friction_patch_boundaries(:,2);

    figure % Slip patch boundaries vs. p_inc
    hold on
    p1 = plot(-p_plot/1e6,y_tilde_1_plot,'-o','Color','r','LineWidth',1.0,'MarkerIndices',...
         1:15:length(y_tilde_1_plot),'MarkerSize',5.0);
    plot(-p_tilde_2_plot/1e6,y_tilde_2_plot,'-o','Color','r','LineWidth',1.0,'MarkerIndices',...
         1:15:length(y_tilde_2_plot),'MarkerSize',5.0);
    plot(-p_tilde_3_plot/1e6,y_tilde_3_plot,'-o','Color','r','LineWidth',1.0,'MarkerIndices',...
         1:15:length(y_tilde_3_plot),'MarkerSize',5.0);
    plot(-p_plot/1e6,y_tilde_4_plot,'-o','Color','r','LineWidth',1.0,'MarkerIndices',...
         1:15:length(y_tilde_4_plot),'MarkerSize',5.0);
    p2 = plot(p_DARTS,y_tilde_DARTS,'-','Color',blue,'LineWidth',1.0);
    line([ 0 35],[   0    0],'LineStyle','-','Color','k','LineWidth',0.5) % horizontal line
    line([ 0 35],[   a    a],'LineStyle',':','Color','k','LineWidth',0.5)
    line([ 0 35],[  -a   -a],'LineStyle',':','Color','k','LineWidth',0.5)
    line([26.93 26.93],[y_plot_min_bounds y_plot_max_bounds],'LineStyle',':','Color','r',...
        'LineWidth',1.0)
    legend([p1 p2],'Analytical','DARTS','location','west')
    if flag_plot_slip_weakening_friction == 0
        xlabel('$\it -p$ (MPa)','Interpreter','latex','FontSize',11)
        ylabel('$\it \tilde{y}_i$ (m)','Interpreter','latex','FontSize',11)
        axis([-p_max_bounds/1e6 -p_min_bounds/1e6 y_plot_min_bounds y_plot_max_bounds])
        box on
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Part 3: Slip-weakening friction:
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if flag_plot_slip_weakening_friction == 1
    if flag_plot_constant_friction == 0
        figure % Slip patch boundaries vs. p_inc
        hold on
        line([ 0 35],[   0    0],'LineStyle','-','Color','k','LineWidth',0.5) % horizontal line
        line([ 0 35],[   a    a],'LineStyle',':','Color','k','LineWidth',0.5)
        line([ 0 35],[  -a   -a],'LineStyle',':','Color','k','LineWidth',0.5)
    end
    flag_fric = 2;
    mu_dyn = 0.40; % dynamic friction coefficient, -
    flag_cheb = 1; % semi-analytical integration with Chebyshev polynomials
    flag_eig_use = 2; % numerical derivative used for seismicity detection 
    flag_lag = 0; % lagging switched off
    Delta_y_int = 0.01; % integration step size, m
    tol_rel_coupled = 1e-12; % relative tolerance for for patch size computation
    Delta_p_min = -50; % minimum allowed pressure step size (negative for production), Pa
    Delta_y_target = 0.05; % target change in y value during pressure stepping, m
    p_inc_01 = -13e6; % incremental pressure at which to take a snapshot, Pa
    p_inc_02 = -14e6;
    p_inc_03 = -15e6;
    p_inc_04 = -16e6;
    p_inc_05 = -17e6;
    flag_plot_snap = 0; % flag for snapshot plots; plotting switched off
    flag_plot_eig = 0; % flag for eigenvalue plot; plotting switched off
    [~,~,~,~,~,p_plot,~,~,p_seismic,p_tilde_2_plot,p_tilde_3_plot,~,~,~,~,~,~,...
        y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,~,~] = ...
        fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
        D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
        eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
        flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
        flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
        g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
        p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
        p_max_bounds,p_min_bounds,...
        phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
        y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
        y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
        y_plot_n_pt);
    
    % Create second part of plot:
    plot(-p_plot(1:end-1)/1e6,y_tilde_1_plot(1:end-1),'-','Color',orange,'LineWidth',1.5)
    plot(-p_tilde_2_plot(1:end-1)/1e6,y_tilde_2_plot(1:end-1),'-','Color',orange,...
        'LineWidth',1.5)
    plot(-p_tilde_3_plot(1:end-1)/1e6,y_tilde_3_plot(1:end-1),'-','Color',orange,...
        'LineWidth',1.5)
    plot(-p_plot(1:end-1)/1e6,y_tilde_4_plot(1:end-1),'-','Color',orange,'LineWidth',1.5)
    line([-p_seismic/1e6 -p_seismic/1e6],[y_plot_min_bounds y_plot_max_bounds],...
        'LineStyle',':','Color',orange,'LineWidth',1.0)
    text(-p_seismic/1e6+0.3,max(y_tilde_1_plot(end-1),y_plot_min_bounds)-6,...
        '$\it \tilde{y}_1$','Interpreter','latex','FontSize',11)
    text(-p_seismic/1e6+0.3,y_tilde_2_plot(end-1)+7,'$\it \tilde{y}_2$','Interpreter',...
        'latex','FontSize',11)
    text(-p_seismic/1e6+0.3,y_tilde_3_plot(end-1)-5,'$\it \tilde{y}_3$','Interpreter',...
        'latex','FontSize',11)
    text(-p_seismic/1e6+0.3,min(y_tilde_4_plot(end-1),y_plot_max_bounds)+7,...
        '$\it \tilde{y}_4$','Interpreter','latex','FontSize',11)
    
    % Repeat for lower mu_dyn:
    mu_dyn = 0.20; % dynamic friction coefficient, -
    [~,~,~,~,~,p_plot,~,~,p_seismic,p_tilde_2_plot,p_tilde_3_plot,~,~,~,~,~,~,...
        y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,~,~] = ...
        fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
        D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
        eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
        flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
        flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
        g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
        p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
        p_max_bounds,p_min_bounds,...
        phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
        y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
        y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
        y_plot_n_pt);
    
    % Create third part of plot:
    plot(-p_plot(1:end-1)/1e6,y_tilde_1_plot(1:end-1),'-','Color','red','LineWidth',1.5)
    plot(-p_tilde_2_plot(1:end-1)/1e6,y_tilde_2_plot(1:end-1),'-','Color','red',...
        'LineWidth',1.5)
    plot(-p_tilde_3_plot(1:end-1)/1e6,y_tilde_3_plot(1:end-1),'-','Color','red',...
        'LineWidth',1.5)
    plot(-p_plot(1:end-1)/1e6,y_tilde_4_plot(1:end-1),'-','Color','red','LineWidth',1.5)   
    line([ 0 35],[   0    0],'LineStyle','-','Color','k','LineWidth',0.5) % horizontal line
    line([ 0 35],[   a    a],'LineStyle',':','Color','k','LineWidth',0.5)
    line([ 0 35],[  -a   -a],'LineStyle',':','Color','k','LineWidth',0.5)
    line([-p_seismic/1e6 -p_seismic/1e6],[y_plot_min_bounds y_plot_max_bounds],...
        'LineStyle',':','Color','red','LineWidth',1.0)
    box on
    text(-p_seismic/1e6+0.3,max(y_tilde_1_plot(end-1),y_plot_min_bounds)-6,...
        '$\it \tilde{y}_1$','Interpreter','latex','FontSize',11)
    text(-p_seismic/1e6+0.3,y_tilde_2_plot(end-1)+7,'$\it \tilde{y}_2$','Interpreter',...
        'latex','FontSize',11)
    text(-p_seismic/1e6+0.3,y_tilde_3_plot(end-1)-5,'$\it \tilde{y}_3$','Interpreter',...
        'latex','FontSize',11)
    text(-p_seismic/1e6+0.3,min(y_tilde_4_plot(end-1),y_plot_max_bounds)+7,...
        '$\it \tilde{y}_4$','Interpreter','latex','FontSize',11)
    xlabel('$\it -p$ (MPa)','Interpreter','latex','FontSize',11)
    ylabel('$\tilde{y}_i$ (m)','Interpreter','latex','FontSize',11)
    axis([-p_max_bounds/1e6 -p_min_bounds/1e6 y_plot_min_bounds y_plot_max_bounds])
    
    % Plot small circle:
    cx_0 = -p_seismic/1e6; 
    cy_0 = 61.0584;
    cr = 0.20;
    asp_rat_1 = 5/4; % default axis ratio (approximately? can't find any Matlab documentation)
    asp_rat_2 = (y_plot_max_bounds-y_plot_min_bounds)/(-p_min_bounds/1e6 - -p_max_bounds/1e6);  
    cphi = linspace(0,pi,180);
    cx_1 = cx_0+cr*sin(cphi);
    cy_1 = cy_0+cr*cos(cphi)*asp_rat_1*asp_rat_2;
    cx_2 = cx_0-cr*sin(cphi);
    cy_2 = cy_1;
    plot(cx_1,cy_1,'LineStyle','-','Color','k','LineWidth',0.75)
    plot(cx_2,cy_2,'LineStyle','-','Color','k','LineWidth',0.75)   
    % Plot larger circle:
    dx_0 = ( (-p_seismic/1e6)+(-p_max_bounds/1e6))/2; 
    dy_0 = 35;
    dr = 1.4;
    asp_rat_2 = (y_plot_max_bounds-y_plot_min_bounds)/(-p_min_bounds/1e6 - -p_max_bounds/1e6);  
    dphi = linspace(0,pi,180);
    dx_1 = dx_0+dr*sin(dphi);
    dy_1 = dy_0+dr*cos(dphi)*asp_rat_1*asp_rat_2;
    dx_2 = dx_0-dr*sin(dphi);
    dy_2 = dy_1;
    plot(dx_1,dy_1,'LineStyle','-','Color','k','LineWidth',0.75)
    plot(dx_2,dy_2,'LineStyle','-','Color','k','LineWidth',0.75)
    % plot magnified version of y_tilde_3 vs. p:
    line([dx_0 dx_0],[dy_1(1) dy_1(end)],...
        'LineStyle',':','Color','red','LineWidth',1.0)
    x_small = -p_tilde_3_plot(1:end-1)/1e6;
    y_small = y_tilde_3_plot(1:end-1);
    x_larger = dx_0 + (x_small-cx_0)*(dr/cr)/2;
    y_larger = dy_0 + (y_small-cy_0)*(dr/cr)/2;
    x_larger_plot = nonzeros(x_larger .* (13.6866 < x_larger));
    y_larger_plot = nonzeros(y_larger .* (13.6866 < x_larger));
    plot(x_larger_plot,y_larger_plot,'-','Color','red','LineWidth',1.5)
    hold off
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Part 4: Slip weakening; parameter sensitivity check
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% if flag_plot_parameter_sensitivity == 1
%     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%     % Slip-weaking friction; 1st series:
%     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%     % 4.1.1:
%     flag_fric = 2;
%     flag_cheb = 0; % numerical integration with near-uniform staggered grids for y and xi
%     flag_eig_use = 0; % eigenvalues used to check accuracy of simulated slip length
%     flag_lag = 0; % lagging switched off
%     Delta_y_int = 0.10; % integration step size, m
%     tol_rel_coupled = 1e-7; % rel. tol. for convergence of coupling and slip weakening
%     Delta_p_min = -0.005e6; % minimum allowed step size (negative for production), Pa
%     Delta_y_target = 0.50; % target change in y value, m
%     flag_plot_diag = 0;
%     flag_plot_eig = 1;
%     flag_plot_snap = 0;
%     p_inc_01 = -24e6; % incremental pressure at which to take a snapshot, Pa
%     p_inc_02 = -25e6;
%     p_inc_03 = -26e6;
%     p_inc_04 = -27e6;
%     p_inc_05 = -28e6;
%     % Note: flag_plot_eig = 1; plotting switched off but data exported to this file through header.
%     [Delta_y_plot_top_01,~,e_num_plot_01,~,~,~,p_plot_01] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     % 4.1.2:
%     tol_rel_coupled = 1e-7; % rel. tol. for convergence of coupling and slip weakening
%     Delta_p_min = -0.05e6; % minimum allowed step size (negative for production), Pa
%     [Delta_y_plot_top_02,~,e_num_plot_02,~,~,p_plot_02] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     % 4.1.3:
%     tol_rel_coupled = 1e-7; % rel. tol. for convergence of coupling and slip weakening
%     Delta_p_min = -0.5e6; % minimum allowed step size (negative for production), Pa
%     [Delta_y_plot_top_03,~,e_num_plot_03,~,~,p_plot_03] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     % 4.1.4: Plot slip patch nucleation length as computed from simulation and eigen values:
%     figure
%     subplot(3,1,1)
%     hold on
%     plot(-p_plot_01(2:end-1)/1e6,Delta_y_plot_top_01(2:end-1),'-','MarkerSize',4,...
%         'Color','r','LineWidth',1.5);
%     plot(-p_plot_01(2:end-1)/1e6,1./e_num_plot_01(2:end-1),'-','MarkerSize',4,...
%         'Color',blue,'LineWidth',1.5);
%     plot(-p_plot_02(2:end-1)/1e6,Delta_y_plot_top_02(2:end-1),'--x','MarkerSize',5,...
%         'Color','r','LineWidth',1.5);
%     plot(-p_plot_02(2:end-1)/1e6,1./e_num_plot_02(2:end-1),'--x','MarkerSize',5,...
%         'Color',blue,'LineWidth',1.5);
%     plot(-p_plot_03(2:end-1)/1e6,Delta_y_plot_top_03(2:end-1),':o','MarkerSize',2,...
%         'Color','r','LineWidth',1.5);
%     plot(-p_plot_03(2:end-1)/1e6,1./e_num_plot_03(2:end-1),':o','MarkerSize',2,...
%         'Color',blue,'LineWidth',1.5);
%     plot(-p_plot_01(end-1)/1e6,Delta_y_plot_top_01(end-1),'p','MarkerSize',12,...
%         'MarkerEdgeColor',...
%         'r','MarkerFaceColor','r');
%     plot(-p_plot_01(end-1)/1e6,1./e_num_plot_01(end-1),'^','MarkerSize',9,'MarkerEdgeColor',...
%         blue,'MarkerFaceColor',blue);
%     plot(-p_plot_02(end-1)/1e6,Delta_y_plot_top_02(end-1),'p','MarkerSize',12,...
%         'MarkerEdgeColor',...
%         'r','MarkerFaceColor','r');
%     plot(-p_plot_02(end-1)/1e6,1./e_num_plot_02(end-1),'^','MarkerSize',9,...
%         'MarkerEdgeColor',...
%         blue,'MarkerFaceColor',blue);
%     plot(-p_plot_03(end-1)/1e6,Delta_y_plot_top_03(end-1),'p','MarkerSize',12,...
%         'MarkerEdgeColor',...
%         'r','MarkerFaceColor','r');
%     plot(-p_plot_03(end-1)/1e6,1./e_num_plot_03(end-1),'^','MarkerSize',9,...
%         'MarkerEdgeColor',...
%         blue,'MarkerFaceColor',blue);
%     plot(-p_plot_01(end-1)/1e6,Delta_y_plot_top_01(end-1),'o','MarkerSize',15,...
%         'MarkerEdgeColor',...
%         green,'MarkerFaceColor','none');
%     xlabel('$\it -p$ (MPa)','Interpreter','latex','FontSize',11)
%     ylabel('$\it \Delta y^*$ (m)','Interpreter','latex','FontSize',11)
%     axis([16.9 18.6 10 35])
%     box on
%     grid on
%     legend...
%         ('$\Delta {\it y^*_{sim}},\Delta {\it p_{min}} = -0.005\: {\rm MPa}, {\it tol} = 10^{-7}$',...
%         '$\Delta {\it y^*_{eig}}, \hspace{2.2 pt} \Delta {\it p_{min}} = -0.005\: {\rm MPa}, {\it tol} = 10^{-7}$',...
%         '$\Delta {\it y^*_{sim}}, \Delta {\it p_{min}} = \hspace{5.4 pt} -0.05\: {\rm MPa}, {\it tol} = 10^{-7}$',...
%         '$\Delta {\it y^*_{eig}}, \hspace{2.2 pt} \Delta {\it p_{min}} = \hspace{5.4 pt} -0.05\: {\rm MPa}, {\it tol} = 10^{-7}$',...
%         '$\Delta {\it y^*_{sim}}, \Delta {\it p_{min}} = \hspace{11 pt} -0.5\: {\rm MPa}, {\it tol} = 10^{-7}$',...
%         '$\Delta {\it y^*_{eig}}, \hspace{2.2 pt} \Delta {\it p_{min}} = \hspace{11 pt} -0.5\: {\rm MPa}, {\it tol} = 10^{-7}$',...
%         'Interpreter','latex','FontSize',12,'location','east')
%     
%     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%     % Slip-weaking friction; 2nd series:
%     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%     % 4.2.1:
%     tol_rel_coupled = 1e-7; % rel. tol. for convergence of coupling and slip weakening
%     Delta_p_min = -0.005e6; % minimum allowed step size (negative for production), Pa
%     [Delta_y_plot_top_01,~,e_num_plot_01,~,~,p_plot_01] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     % 4.2.2:
%     tol_rel_coupled = 1e-5; % rel. tol. for convergence of coupling and slip weakening
%     Delta_p_min = -0.005e6; % minimum allowed step size (negative for production), Pa
%     [Delta_y_plot_top_02,~,e_num_plot_02,~,p_plot_02] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     % 4.2.3:
%     tol_rel_coupled = 1e-3; % rel. tol. for convergence of coupling and slip weakening
%     Delta_p_min = -0.005e6; % minimum allowed step size (negative for production), Pa
%     [Delta_y_plot_top_03,~,e_num_plot_03,~,~,p_plot_03] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     % 4.2.4: Plot slip patch nucleation length as computed from simulation and eigen values:
%     subplot(3,1,2)
%     hold on
%     plot(-p_plot_01(2:end-1)/1e6,Delta_y_plot_top_01(2:end-1),'-','MarkerSize',4,...
%         'Color','r','LineWidth',1.5);
%     plot(-p_plot_01(2:end-1)/1e6,1./e_num_plot_01(2:end-1),'-','MarkerSize',4,'Color',blue,'LineWidth',1.5);
%     plot(-p_plot_02(2:end-1)/1e6,Delta_y_plot_top_02(2:end-1),'--x','MarkerSize',5,...
%         'Color','r','LineWidth',1.5);
%     plot(-p_plot_02(2:end-1)/1e6,1./e_num_plot_02(2:end-1),'--x','MarkerSize',5,...
%         'Color',blue,'LineWidth',1.5);
%     plot(-p_plot_03(2:end-1)/1e6,Delta_y_plot_top_03(2:end-1),':o','MarkerSize',2,...
%         'Color','r','LineWidth',1.5);
%     plot(-p_plot_03(2:end-1)/1e6,1./e_num_plot_03(2:end-1),':o','MarkerSize',2,...
%         'Color',blue,'LineWidth',1.5);
%     plot(-p_plot_01(end-1)/1e6,Delta_y_plot_top_01(end-1),'p','MarkerSize',12,...
%         'MarkerEdgeColor',...
%         'r','MarkerFaceColor','r');
%     plot(-p_plot_01(end-1)/1e6,1./e_num_plot_01(end-1),'^','MarkerSize',9,...
%         'MarkerEdgeColor',...
%         blue,'MarkerFaceColor',blue);
%     plot(-p_plot_02(end-1)/1e6,Delta_y_plot_top_02(end-1),'p','MarkerSize',12,...
%         'MarkerEdgeColor',...
%         'r','MarkerFaceColor','r');
%     plot(-p_plot_02(end-1)/1e6,1./e_num_plot_02(end-1),'^','MarkerSize',9,'MarkerEdgeColor',...
%         blue,'MarkerFaceColor',blue);
%     plot(-p_plot_03(end-1)/1e6,Delta_y_plot_top_03(end-1),'p','MarkerSize',12,...
%         'MarkerEdgeColor',...
%         'r','MarkerFaceColor','r');
%     plot(-p_plot_03(end-1)/1e6,1./e_num_plot_03(end-1),'^','MarkerSize',9,...
%         'MarkerEdgeColor',...
%         blue,'MarkerFaceColor',blue);
%     plot(-p_plot_01(end-1)/1e6,Delta_y_plot_top_01(end-1),'o','MarkerSize',15,...
%         'MarkerEdgeColor',...
%         green,'MarkerFaceColor','none');
%     xlabel('$\it -p$ (MPa)','Interpreter','latex','FontSize',11)
%     ylabel('$\it \Delta y^*$ (m)','Interpreter','latex','FontSize',11)
%     axis([16.9 18.6 10 35])
%     box on
%     grid on
%     legend...
%         ('$\Delta {\it y^*_{sim}},\Delta {\it p_{min}} = -0.005\: {\rm MPa}, {\it tol} = 10^{-7}$',...
%         '$\Delta {\it y^*_{eig}}, \hspace{2.2 pt} \Delta {\it p_{min}} = -0.005\: {\rm MPa}, {\it tol} = 10^{-7}$',...
%         '$\Delta {\it y^*_{sim}}, \Delta {\it p_{min}} = -0.005\: {\rm MPa}, {\it tol} = 10^{-5}$',...
%         '$\Delta {\it y^*_{eig}}, \hspace{2.2 pt} \Delta {\it p_{min}} = -0.005\: {\rm MPa}, {\it tol} = 10^{-5}$',...
%         '$\Delta {\it y^*_{sim}}, \Delta {\it p_{min}} = -0.005\: {\rm MPa}, {\it tol} = 10^{-3}$',...
%         '$\Delta {\it y^*_{eig}}, \hspace{2.2 pt} \Delta {\it p_{min}} = -0.005\: {\rm MPa}, {\it tol} = 10^{-3}$',...
%         'Interpreter','latex','FontSize',12,'location','east')
%     
%     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%     % Slip-weaking friction; 3rd series:
%     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%     % 4.3.1:
%     tol_rel_coupled = 1e-7; % rel. tol. for convergence of coupling and slip weakening
%     Delta_p_min = -0.005e6; % minimum allowed step size (negative for production), Pa
%     [Delta_y_plot_top_01,~,e_num_plot_01,~,~,p_plot_01] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     % 4.3.2:
%     tol_rel_coupled = 1e-5; % rel. tol. for convergence of coupling and slip weakening
%     Delta_p_min = -0.05e6; % minimum allowed step size (negative for production), Pa
%     [Delta_y_plot_top_02,~,e_num_plot_02,~,p_plot_02] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     % 4.3.3:
%     tol_rel_coupled = 1e-3; % rel. tol. for convergence of coupling and slip weakening
%     Delta_p_min = -0.5e6; % minimum allowed step size (negative for production), Pa
%     [Delta_y_plot_top_03,~,e_num_plot_03,~,~,p_plot_03] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     % 4.3.4: Plot slip patch nucleation length as computed from simulation and eigen values:
%     subplot(3,1,3)
%     hold on
%     plot(-p_plot_01(2:end-1)/1e6,Delta_y_plot_top_01(2:end-1),'-','MarkerSize',4,...
%         'Color','r','LineWidth',1.5);
%     plot(-p_plot_01(2:end-1)/1e6,1./e_num_plot_01(2:end-1),'-','MarkerSize',4,...
%         'Color',blue,'LineWidth',1.5);
%     plot(-p_plot_02(2:end-1)/1e6,Delta_y_plot_top_02(2:end-1),'--x','MarkerSize',5,...
%         'Color','r','LineWidth',1.5);
%     plot(-p_plot_02(2:end-1)/1e6,1./e_num_plot_02(2:end-1),'--x','MarkerSize',5,...
%         'Color',blue,'LineWidth',1.5);
%     plot(-p_plot_03(2:end-1)/1e6,Delta_y_plot_top_03(2:end-1),':o','MarkerSize',2,...
%         'Color','r','LineWidth',1.5);
%     plot(-p_plot_03(2:end-1)/1e6,1./e_num_plot_03(2:end-1),':o','MarkerSize',2,...
%         'Color',blue,'LineWidth',1.5);
%     plot(-p_plot_01(end-1)/1e6,Delta_y_plot_top_01(end-1),'p','MarkerSize',12,...
%         'MarkerEdgeColor',...
%         'r','MarkerFaceColor','r');
%     plot(-p_plot_01(end-1)/1e6,1./e_num_plot_01(end-1),'^','MarkerSize',9,...
%         'MarkerEdgeColor',...
%         blue,'MarkerFaceColor',blue);
%     plot(-p_plot_02(end-1)/1e6,Delta_y_plot_top_02(end-1),'p','MarkerSize',12,...
%         'MarkerEdgeColor',...
%         'r','MarkerFaceColor','r');
%     plot(-p_plot_02(end-1)/1e6,1./e_num_plot_02(end-1),'^','MarkerSize',9,...
%         'MarkerEdgeColor',...
%         blue,'MarkerFaceColor',blue);
%     plot(-p_plot_03(end-1)/1e6,Delta_y_plot_top_03(end-1),'p','MarkerSize',12,...
%         'MarkerEdgeColor',...
%         'r','MarkerFaceColor','r');
%     plot(-p_plot_03(end-1)/1e6,1./e_num_plot_03(end-1),'^','MarkerSize',9,...
%         'MarkerEdgeColor',...
%         blue,'MarkerFaceColor',blue);
%     plot(-p_plot_01(end-1)/1e6,Delta_y_plot_top_01(end-1),'o','MarkerSize',15,...
%         'MarkerEdgeColor',...
%         green,'MarkerFaceColor','none');
%     xlabel('$\it -p$ (MPa)','Interpreter','latex','FontSize',11)
%     ylabel('$\it \Delta y^*$ (m)','Interpreter','latex','FontSize',11)
%     axis([16.9 19.1 10 35])
%     box on
%     grid on
%     lgd = legend...
%         ('$\Delta {\it y^*_{sim}},\Delta {\it p_{min}} = -0.005\: {\rm MPa}, {\it tol} = 10^{-7}$',...
%         '$\Delta {\it y^*_{eig}}, \hspace{2.2 pt} \Delta {\it p_{min}} = -0.005\: {\rm MPa}, {\it tol} = 10^{-7}$',...
%         '$\Delta {\it y^*_{sim}}, \Delta {\it p_{min}} = \hspace{5.4 pt} -0.05\: {\rm MPa}, {\it tol} = 10^{-5}$',...
%         '$\Delta {\it y^*_{eig}}, \hspace{2.2 pt} \Delta {\it p_{min}} = \hspace{5.4 pt} -0.05\: {\rm MPa}, {\it tol} = 10^{-5}$',...
%         '$\Delta {\it y^*_{sim}}, \Delta {\it p_{min}} = \hspace{11 pt} -0.5\: {\rm MPa}, {\it tol} = 10^{-3}$',...
%         '$\Delta {\it y^*_{eig}}, \hspace{2.2 pt} \Delta {\it p_{min}} = \hspace{11 pt} -0.5\: {\rm MPa}, {\it tol} = 10^{-3}$',...
%         'Interpreter','latex','FontSize',12,'location','north');
%     lgd.NumColumns = 2;
%     hold on
% end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Part 5: Friction coefficient sensitivity check
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% if flag_plot_friction_sensitivity == 1
%     flag_fric = 1;
%     mu_st = 0.30; % static friction coefficient, -
%     Delta_y_int = 0.10; % integration step size, m
%     tol_rel_coupled = 1e-4; % relative tolerance for convergence of coupling and slip weakening
%     p_max_bounds = 0; % highest incremental pressure (negative for production), Pa
%     p_min_bounds = -25e6; % lowest incremental pressure (negative for production), Pa
%     Delta_p_max = -0.1e6; % maximum allowed step size (negative for production), Pa %%%%
%     Delta_p_min = -0.01e6; % minimum allowed pressure step size (negative for production), Pa
%     Delta_p_init = -0.05e6; % initial pressure step size (negative for production), Pa
%     Delta_y_target = 0.10; % target change in y value during pressure stepping, m
%     p_inc_01 = -24e6; % incremental pressure at which to take a snapshot, Pa
%     p_inc_02 = -25e6;
%     p_inc_03 = -26e6;
%     p_inc_04 = -27e6;
%     p_inc_05 = -28e6;
%     flag_echo = 2; % full echo
%     flag_plot_diag = 0;
%     flag_plot_eig = 0; % flag for eigenvalue plot; plotting switched off
%     flag_plot_snap = 0; % flag for snapshot plots; plotting switched on
%     y_plot_max_bounds =  150; % covers both slip patches (overview), m
%     y_plot_min_bounds = -150; % covers both slip patches (overview), m   
   
%     [~,~,~,~,~,p_plot,~,~,~,p_tilde_2_plot,p_tilde_3_plot,...
%         ~,~,~,~,~,~,y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,~,~] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     figure % Slip patch boundaries vs. p_inc
%     hold on
%     line([ 0 35],[   0    0],'LineStyle','-','Color','k','LineWidth',0.5) % horizontal line
%     line([ 0 35],[   a    a],'LineStyle',':','Color','k','LineWidth',0.5)
%     line([ 0 35],[  -a   -a],'LineStyle',':','Color','k','LineWidth',0.5)
%     plot(-p_plot/1e6,y_tilde_1_plot,'-','Color',blue,'LineWidth',1.5)
%     plot(-p_tilde_2_plot/1e6,y_tilde_2_plot,'-','Color',blue,'LineWidth',1.5)
%     plot(-p_tilde_3_plot/1e6,y_tilde_3_plot,'-','Color',blue,'LineWidth',1.5)
%     plot(-p_plot/1e6,y_tilde_4_plot,'-','Color',blue,'LineWidth',1.5)
%     
%     flag_fric = 1;
%     mu_st = 0.40; % static friction coefficient, -
%     Delta_p_init = -0.05e6; % initial pressure step size (negative for production), Pa
%     Delta_y_target = 0.20; % target change in y value during pressure stepping, m
%     [~,~,~,~,~,p_plot,~,~,~,p_tilde_2_plot,p_tilde_3_plot,...
%         ~,~,~,~,~,~,y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,~,~] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     plot(-p_plot/1e6,y_tilde_1_plot,'-','Color',blue,'LineWidth',1.5)
%     plot(-p_tilde_2_plot/1e6,y_tilde_2_plot,'-','Color',blue,'LineWidth',1.5)
%     plot(-p_tilde_3_plot/1e6,y_tilde_3_plot,'-','Color',blue,'LineWidth',1.5)
%     plot(-p_plot/1e6,y_tilde_4_plot,'-','Color',blue,'LineWidth',1.5)
%     
%     flag_fric = 2;
%     mu_st = 0.52; % static friction coefficient, -
%     mu_dyn = 0.30; % dynamic friction coefficient, -
%     p_max_bounds = 0; % highest incremental pressure (negative for production), Pa
%     flag_cheb = 1; % semi-analytical integration with Chebyshev polynomials
%     flag_eig_use = 1; % eigenvalues used for seismicity detection
%     flag_lag = 0; % lagging switched off
%     Delta_y_int = 0.01; % integration step size, m
%     tol_rel_coupled = 1e-7; % rel. tol. for convergence of coupling and slip weakening
%     Delta_p_min = -0.005e6; % minimum allowed step size (negative for production), Pa
%     Delta_y_target = 0.10; % target change in y value, m
%     [~,~,~,~,~,p_plot,~,~,p_seismic,p_tilde_2_plot,p_tilde_3_plot,~,~,~,~,~,~,...
%         y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,~,~] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     line([-p_seismic/1e6 -p_seismic/1e6],[y_plot_min_bounds y_plot_max_bounds],...
%         'LineStyle',':','Color','r','LineWidth',0.5)
%     plot(-p_plot(1:end-1)/1e6,y_tilde_1_plot(1:end-1),'-','Color','red','LineWidth',1.5)
%     plot(-p_tilde_2_plot(1:end-1)/1e6,y_tilde_2_plot(1:end-1),'-','Color','red',...
%         'LineWidth',1.5)
%     plot(-p_tilde_3_plot(1:end-1)/1e6,y_tilde_3_plot(1:end-1),'-','Color','red',...
%         'LineWidth',1.5)
%     plot(-p_plot(1:end-1)/1e6,y_tilde_4_plot(1:end-1),'-','Color','red','LineWidth',1.5)    
%    
%     flag_fric = 2;
%     mu_st = 0.52; % static friction coefficient, -
%     mu_dyn = 0.40; % dynamic friction coefficient, -
%     p_max_bounds = -10e6; % highest incremental pressure (negative for production), Pa
%     [~,~,~,~,~,p_plot,~,~,p_seismic,p_tilde_2_plot,p_tilde_3_plot,~,~,~,~,~,~,...
%         y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,~,~] = ...
%         fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
%         D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
%         eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
%         flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
%         flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
%         g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
%         p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
%         p_max_bounds,p_min_bounds,...
%         phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
%         y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
%         y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
%         y_plot_n_pt);
%     
%     line([-p_seismic/1e6 -p_seismic/1e6],[y_plot_min_bounds y_plot_max_bounds],...
%         'LineStyle',':','Color','r','LineWidth',0.5)
%     plot(-p_plot(1:end-1)/1e6,y_tilde_1_plot(1:end-1),'-','Color','red','LineWidth',1.5)
%     plot(-p_tilde_2_plot(1:end-1)/1e6,y_tilde_2_plot(1:end-1),'-','Color','red',...
%         'LineWidth',1.5)
%     plot(-p_tilde_3_plot(1:end-1)/1e6,y_tilde_3_plot(1:end-1),'-','Color','red',...
%         'LineWidth',1.5)
%     plot(-p_plot(1:end-1)/1e6,y_tilde_4_plot(1:end-1),'-','Color','red','LineWidth',1.5)
%     text(25.3,-150,'$\it \tilde{y}_1, \mu_{st} = 0.3$','Interpreter','latex','FontSize',11)
%     text(25.3,-118,'$\it \tilde{y}_1, \mu_{st} = 0.4$','Interpreter','latex','FontSize',11)
%     text(25.3, 120,'$\it \tilde{y}_4, \mu_{st} = 0.4$','Interpreter','latex','FontSize',11)
%     text(25.3, 150,'$\it \tilde{y}_4, \mu_{st} = 0.3$','Interpreter','latex','FontSize',11)
%     text(21.5,-140,'$\it \mu_{st}\rm = 0.3$','Interpreter','latex','FontSize',11)
%     text(21.5,-113,'$\it \mu_{st}\rm = 0.4$','Interpreter','latex','FontSize',11)
%     text(21.5, 117,'$\it \mu_{st}\rm = 0.4$','Interpreter','latex','FontSize',11)
%     text(21.5, 140,'$\it \mu_{st}\rm = 0.3$','Interpreter','latex','FontSize',11)
%     x_1 = [18.95/25  18.95/25];
%     y_1 = [97.22/150 150/150];
%     annotation('arrow',x_1,y_1)
%     xlabel('$\it -p$ (MPa)','Interpreter','latex','FontSize',11)
%     ylabel('$\it y_i, \tilde{y}_i$ (m)','Interpreter','latex','FontSize',11)
%     axis([0 25 y_plot_min_bounds y_plot_max_bounds])
%     box on
% end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Part 6: Fault offset sensitivity check
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if flag_plot_offset_sensitivity == 1
    % -----------------------------------------------------------------------------------------
    % Set input parameters:
    % -----------------------------------------------------------------------------------------
    % Parameter set for Example 1 from paper:
    alpha = 0.9; % Biot's coefficient, -
    delta_c = 0.02; % critical slip distance for slip weakening 0, m
    D_center = 3500; % reservoir depth @ y = 0, m
    flag_fric = 2; % slip-weakening friction
    g = 9.81; % acceleration of gravity, m/s^2
    G = 6500e6; % shear modulus, Pa
    kappa = 0; % cohesion, -
    K_eff = 0.5; % initial effective vertical/horizontal stress ratio, -
%     mu_dyn = 0.20; % dynamic friction coefficient 0, -
    mu_dyn = 0.30; % dynamic friction coefficient 0, -
    mu_st = 0.52; % static friction coefficient, -
    nu = 0.15; % Poisson's ratio, -
    p_0_center = 35e6; % initial pressure @ y = 0, Pa
    phi = 0.15; % porosity, -
    rho_f = 1020; % fluid density, kg/m^3
    rho_s = 2650; % solid density, kg/m^3
    sigma_v_0 = 0; % initial vertical stress @ y = 0, Pa. If set equal to zero, the stress ...
        % is computed from the weigth of the overburden; see function fault_slip_sigma.m.
    theta = 70*pi/180; % dip angle, rad
    
    % -----------------------------------------------------------------------------------------
    % Set program flow flags and computational parameters:
    % -----------------------------------------------------------------------------------------
    % Program flow flags:
    flag_cheb = 1; % semi-analytical integration with Chebyshev polynomials
    flag_cor = 1; % forcing switched on (recommended for regular use)
    flag_decouple = 0; % coupled simulation
    flag_echo = 1; % limited echo; only info about main simulation components
    flag_eig_use = 1; % stops simulation when Delta y*_sim > Delta y*_eig
    flag_lag = 0; % lagging switched off (recommended for regular use)
    flag_M = 1; % seismic moment computation switched on     
    flag_paper = 1; % plot for paper
    flag_reg = 1; % regularization switched on (recommended for regular use)
    eps_reg = 0; % regularization parameter for slip-dependent friction coefficient, m
    % eps_reg = 2e-4; % regularization parameter for slip-dependent friction coefficient, m
%     eps_reg = 2e-3; % regularization parameter for slip-dependent friction coefficient, m
    eta_reg = 0.10; % regularization parameter for Coulomb stresses, m
%     eta_reg = 0.50; % regularization parameter for Coulomb stresses, m
    
    % Computational parameters for patch size computation:
    Delta_y_int = 0.01; % spatial integration step size, m. Only relevant for flag_cheb = 0
    tol_rel_coupled = 1e-8; % relative tolerance for for patch size computation
    n_max_int = 2000; % max. allowed total nr. of integr. pnts, -. Only rel. for flag_cheb = 0
    n_min_int = 100;  % min. required total nr. of integr. pnts, -. Only rel. for flag_cheb = 0
%    cheb_terms = 100; % number of terms in Chebyshev polynomial approximation
    cheb_terms = 200; % number of terms in Chebyshev polynomial approximation
    max_iter_coupled = 500; % maximum allowed number of iterations for patch size computation   
    
    % Computational parameters for pressure stepping:
    p_max_bounds = -5e6; % highest incremental pressure (negative for production), Pa
    p_min_bounds = -30e6; % lowest incremental pressure (negative for production), Pa
    Delta_p_init = -0.5e6; % initial pressure step size (negative for production), Pa
    Delta_p_max = -1e6; % maximum allowed pressure step size (negative for production), Pa
    Delta_p_min = -100; % minimum allowed pressure step size (negative for production), Pa
    Delta_y_target = 0.10; % target change in y value during pressure stepping, m
    small_patch = 0.05; % minimum patch size to trigger slip computation during pr. stepping, m
        
    
    % -----------------------------------------------------------------------------------------
    % Set plotting flags and parameters:
    % -----------------------------------------------------------------------------------------
    % Plotting flags:
    flag_plot_cap = 0;
    flag_plot_diag = 0; 
    flag_plot_eig = 0; 
    flag_plot_snap = 0;
    p_inc_01 = -13e6; % incremental pressure at which to take a snapshot, Pa
    p_inc_02 = -14e6;
    p_inc_03 = -15e6;
    p_inc_04 = -16e6;
    p_inc_05 = -17e6;
    % y_plot_max_delta_1 =  250; % covers both slip patches (overview), m
    % y_plot_min_delta_1 = -250; % covers both slip patches (overview), m
    y_plot_max_delta_1 =  100; % covers both slip patches (in detail), m
    y_plot_min_delta_1 = -100; % covers both slip patches (in detail), m
    % y_plot_max_delta_2 =   80; % covers top patch only (further detail), m
    % y_plot_min_delta_2 =   60; % covers top patch only (further detail), m
    y_plot_max_delta_2 =   77; % covers top patch only (further detail), m
    y_plot_min_delta_2 =   65; % covers top patch only (further detail), m
    y_plot_max_sigma   =  250; % covers both slip patches (overview), m
    y_plot_min_sigma   = -250; % covers both slip patches (overview), m
    y_plot_n_pt = 2000; % number of points between y_plot_min_... and y_plot_max_...
    
    % Parameters for plots of Coulomb stress zeros and slip patch boundaries as a function ...
        % of depletion pressure:
    % y_plot_max_bounds =  150; % covers both slip patches (overview), m
    % y_plot_min_bounds = -150; % covers both slip patches (overview), m
    y_plot_max_bounds =  100; % covers both slip patches (in detail), m
    y_plot_min_bounds = -100; % covers both slip patches (in detail), m
    
    % Colors for line plots:
    blue = [0/255 128/255 255/255];
    gray = [235/255 235/255 235/255];
    green = [0/255 204/255 0/255];
    orange = [255/255 185/255 0/255];
    colors = [blue; gray; green; orange];
    
    % -----------------------------------------------------------------------------------------
    % End of input. Start computing:
    % -----------------------------------------------------------------------------------------
    Delta_t_f = 2.5; % half stepsize for plots of fault offset t vs p* and Delta_y*, m
%     Delta_t_f = 4.5; % half stepsize for plots of fault offset t vs p* and Delta_y*, m
    J_plot = 45;
%    J_plot = 24;
    I_rep = 1; % number of repetitions to obtain stable average simulation time 
    t_comp_plot = zeros(J_plot,1); % elapsed time
    for i_rep = 1:I_rep % repeat I_rep times to obtain average simulation time
        t_f_plot = zeros(J_plot,1); % fault throw
        p_inc_nuc_plot = zeros(J_plot,1);
        Delta_y_plot_top_2 = zeros(J_plot,1);
        Delta_y_eig_plot = zeros(J_plot,1);
        Delta_y_UR_plot_2 = zeros(J_plot,1);
        M_a_nuc_plot = zeros(J_plot,1);
        M_s_plot = zeros(J_plot,1);
        M_s_hat_plot = zeros(J_plot,1);
        a = 112.5 + Delta_t_f;
        b = 112.5 - Delta_t_f;
        for j_plot = 1:J_plot
            tic
            a = a - Delta_t_f;
            b = b + Delta_t_f;
            h = a + b; % reservoir height
            t_f = b - a; % fault throw
            t_f_plot(j_plot) = t_f; % fault throw
            [i_rep j_plot t_f_plot(j_plot) a b] % !!! writes to screen to check progress
            [Delta_y_plot_top,Delta_y_UR_plot,e_plot,M_a_plot,p_inc,...
                ~,~,~,~,~,~,~,~,~,~,~,~,~,~,~,~,~,~] = ...
                fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
                D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,...
                eps_reg,eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,...
                flag_decouple,flag_echo,flag_eig_use,flag_fric,flag_lag,flag_M,...
                flag_paper,flag_plot_cap,flag_plot_diag,flag_plot_eig,flag_plot_snap,...
                flag_reg,flag_top_only,g,G,kappa,...
                K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
                p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
                p_max_bounds,p_min_bounds,...
                phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
                y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
                y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
                y_plot_n_pt);
            p_inc_nuc_plot(j_plot) = p_inc; % nucleation pressure
            if length(Delta_y_plot_top) > 1 % to avoid crash in first step
                Delta_y_plot_top_2(j_plot) = Delta_y_plot_top(end-1); % nucleation length (simulated)
            end
            Delta_y_eig_plot(j_plot) = 1/e_plot(end); % nucleation length (eigenvalue)
            Delta_y_UR_plot_2(j_plot) = Delta_y_UR_plot(end); % nucleation length (Uenishi & R)
            M_a_nuc_plot(j_plot) = M_a_plot(end); % aseismic moment at nucleation
            % Repeat simulation at p* for constant friction with mu_st_ps = mu_dyn
            if p_inc > p_min_bounds % p_inc < 0!
                mu_st_ps = mu_dyn;
                flag_decouple_ps = 1;
                flag_fric_ps = 1;
                p_max_bounds_ps = 0; % p_inc < 0!
                p_min_bounds_ps = p_inc;
                Delta_p_init_ps = -0.05e6;
                Delta_y_int_ps = 0.10;
                n_max_int_ps = 2000;
                cheb_terms_ps = 200;
                %            tol_rel_coupled_ps = 1e-6;
                tol_rel_coupled_ps = 1e-8;
                %             Delta_p_min_ps = -1e5;
                Delta_p_min_ps = -1e4;
                %            Delta_y_target_ps = 0.50;
                Delta_y_target_ps = 0.20;
                [~,~,~,M_ps_plot,~,~,~,~,~,~,~,~,~,~,~,~,~,~,~,~,~,~,~] = ...
                    fault_slip_patch_bounds(a,alpha,b,cheb_terms_ps,colors,delta_c,...
                    D_center,Delta_p_init_ps,Delta_p_max,Delta_p_min_ps,Delta_y_int_ps,...
                    Delta_y_target_ps,eps_reg,eta_reg,f_damp,flag_cheb,flag_check,...
                    flag_cont,flag_cor,flag_decouple_ps,flag_echo,flag_eig_use,...
                    flag_fric_ps,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
                    flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,g,G,kappa,...
                    K_eff,max_iter_coupled,mu_dyn,mu_st_ps,n_max_int_ps,...
                    n_min_int,nu,p_0_center,...
                    p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
                    p_max_bounds_ps,p_min_bounds_ps,...
                    phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,...
                    tol_rel_coupled_ps,y_plot_max_bounds,y_plot_max_delta_1,...
                    y_plot_max_delta_2,y_plot_max_sigma,y_plot_min_bounds,...
                    y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
                    y_plot_n_pt);
                M_s_plot(j_plot) = M_ps_plot(end) - M_a_plot(end); % seismic moment
                M_s_hat_plot(j_plot) = -(1-2*nu)*alpha*p_inc*h*t_f; % upper bound
            end
            t_comp_plot(j_plot) = t_comp_plot(j_plot) + toc; % execution time
        end
    end   
    f = figure;
%    f.Position = [489   343   560   420]; % left bottom width height; default values
    f.Position = [489   50   560   1.3*420]; % adapted values 
    subplot(4,1,1) % nucleation pressure
    hold on
    plot(t_f_plot(1:4),-p_inc_nuc_plot(1:4)/1e6,'o','MarkerSize',4,'MarkerEdgeColor','r',...
        'MarkerFaceColor','w')
    plot(t_f_plot(5:end),-p_inc_nuc_plot(5:end)/1e6,'o','MarkerSize',4,'MarkerEdgeColor','r',...
        'MarkerFaceColor','r')
    hold off
%     xlabel('$\it t$ (m)','Interpreter','latex','FontSize',11)
    ylabel('$\it -p*$ (MPa)','Interpreter','latex','FontSize',11)
    axis([0 225 -inf inf])
    text(5,12,'$a)$','Interpreter','latex','FontSize',14)
    box on

    subplot(4,1,2) % nucleation length
    hold on
    plot(t_f_plot(5:end),Delta_y_eig_plot(5:end),'^','MarkerSize',4,'MarkerEdgeColor',blue,...
        'MarkerFaceColor',blue) % eigen value based 
    plot(t_f_plot(5:end),Delta_y_UR_plot_2(5:end),'^','MarkerSize',4,'MarkerEdgeColor',green,...
        'MarkerFaceColor',green) % Uenishi & Rice approximation
    plot(t_f_plot(5:end),Delta_y_plot_top_2(5:end),'^','MarkerSize',4,'MarkerEdgeColor','r',...
        'MarkerFaceColor','r') % simulated nucleation length
    hold off
%     xlabel('$\it t_f$ (m)','Interpreter','latex','FontSize',11)
    ylabel('$\Delta{\it y^*}$ (m)','Interpreter','latex','FontSize',11)
%     axis([0 225 -inf inf])
    axis([0 225 20 25])
    text(5,21,'$b)$','Interpreter','latex','FontSize',14)
    box on

    subplot(4,1,3) % nucleation length
    hold on
%     plot(t_f_plot(3:end),M_a_nuc_plot(3:end)/1e9,'o','MarkerSize',4,'MarkerEdgeColor','r',...
%         'MarkerFaceColor','r')
    plot(t_f_plot(5:end),M_s_hat_plot(5:end)/1e9,'o','MarkerSize',4,'MarkerEdgeColor',blue,...
        'MarkerFaceColor',blue)
    plot(t_f_plot(5:end),M_s_plot(5:end)/1e9,'o','MarkerSize',4,'MarkerEdgeColor','r',...
        'MarkerFaceColor','r')
%     plot(t_f_plot(5:end),M_a_nuc_plot(5:end)/1e9,'LineStyle','-','Color','r','LineWidth','1.5')
%     plot(t_f_plot(5:end),M_s_plot(5:end)/1e9,'LineStyle','-','Color',blue,'LineWidth','1.5')
    hold off
%     xlabel('$\it t_f$ (m)','Interpreter','latex','FontSize',11)
    ylabel('$\it M_s$ (GNm/m)','Interpreter','latex','FontSize',11)
    axis([0 225 -inf 320])
%     axis([0 225 45 115])
    text(5,100,'$c)$','Interpreter','latex','FontSize',14)
    box on
  
    subplot(4,1,4) % averaged elapsed time
    bar(t_f_plot,t_comp_plot/i_rep,'EdgeColor',orange,'FaceColor',orange,'BarWidth',0.6)
    xlabel('$\it t_f$ (m)','Interpreter','latex','FontSize',11)
    ylabel('$t_{comp}$ (s)','Interpreter','latex','FontSize',11)
%    axis([0 225 -inf inf])
    axis([0 225 0 50])
    text(5,10,'$d)$','Interpreter','latex','FontSize',14)
    box on   
end    
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Produce some more figures:
%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if flag_plot_some_more == 1
    % Diagnostics plots:
    a = 75; % see Figure 1, m
    alpha = 0.9; % Biot's coefficient, -
    b = 150; % see Figure 1, m
    delta_c = 0.02; % critical slip distance for slip weakening 0, m
    D_center = 3500; % reservoir depth @ y = 0, m
    flag_fric = 2; % flag to indicate friction law
    g = 9.81; % acceleration of gravity, m/s^2
    G = 6500e6; % shear modulus, Pa
    kappa = 0; % cohesion, -
    K_eff = 0.5; % initial effective vertical/horizontal stress ratio, -
    mu_dyn = 0.20; % dynamic friction coefficient, -
    mu_st = 0.52; % static friction coefficient, -
    nu = 0.15; % Poisson's ratio, -
    p_0_center = 35e6; % initial pressure @ y = 0, Pa
    phi = 0.15; % porosity, -
    rho_f = 1020; % fluid density, kg/m^3
    rho_s = 2650; % solid density, kg/m^3
    sigma_v_0 = 0; % initial vertical stress @ y = 0, Pa. If set equal to zero, the stress is ...
    % computed from the weigth of the overburden; see function fault_slip_sigma.m.
    theta = 70*pi/180; % dip angle, rad
    f_damp = 0.5; % damping factor for iterative update of slip-weakening and coupled ...
    % simulation (only relevant for flag_cheb == 1)
    flag_cheb = 1; % flag to trigger use of Chebyshev polynomials for numerical integration
    flag_check = 0; % flag to trigger graphical check of finite difference and Chebyshev results
    flag_cont = 1; % flag to force continuation in case of non-convergence of slip-weakening ...
    flag_cor = 1; % flag to force correct sign on slip gradient (only relevant for flag_cheb = 0)
    flag_decouple = 1; % flag to force decoupling between slip patches (only relevant for ...
    flag_echo = 1; % flag to trigger writing to screen of simulation progress
    flag_eig_use = 1; % flag to determine use of eigenvalues (only relevant for flag_fric > 1)
    tol_deriv = 1e-6; % simulation stopping criterion in terms of (almost) zero scaled ...
    % numerical derivative (dp/dDelta y)*(Delta_y/A), where A = G / (2*pi*(1-nu)), -
    flag_lag = 0; % flag to trigger lagging of eigen problem computation with one presure step;
    % automatically switched on in case of convergence failure in fault_slip_delta(_cheb).m
    flag_M = 0; % flag to compute seismic moment (can be time consuming)
    flag_paper = 0; % flag to trigger plotting of figure for paper. Warning: may take quite ...
    flag_reg = 1; % flag to control regularization of slip-dependent friction coefficient and ...
    % Coulomb stresses:
    eps_reg = 0; % regularization parameter for slip-dependent friction coefficient, m
    eta_reg = 0.10; % regularization parameter for Coulomb stresses, m
    flag_top_only = 1; % flag to force occurrence of seismicity in top patch only (only ...
    % relevant for flag_cheb == 1)
    Delta_y_int = 0.01;
    n_max_int = 5000;
    cheb_terms = 500;
    tol_rel_coupled = 1e-10;
    n_min_int = 100;  % min. required total nr. of integr. pnts, -. Only relevant for flag_cheb = 0
    max_iter_coupled = 500; % maximum allowed number of iterations for patch size computation
    p_max_bounds = -10e6; % highest incremental pressure (negative for production), Pa
    p_min_bounds = -30e6; % lowest incremental pressure (negative for production), Pa
    Delta_p_init = -0.05e6; % initial pressure step size (negative for production), Pa
    Delta_p_max = -1e6; % maximum allowed pressure step size (negative for production), Pa
    Delta_p_min = -50;
    Delta_y_target = 0.05;
    small_patch = 0.05; % minimum patch size to trigger slip computation during pr. stepping, m
    flag_plot_cap = 0; % flag for plot captions (black) and plot file location captions (red)
    flag_plot_diag = 1; % flag for diagnostic plots:
    flag_plot_eig = 1; % flag for eigenvalue plot (only relevant for slip-weakening friction):
    flag_plot_snap = 0; % flag for snapshot plots:
    p_inc_01 = -14e6; % incremental pressure at which to take a snapshot, Pa
    p_inc_02 = -15e6;
    p_inc_03 = -16e6;
    p_inc_04 = -17e6;
    p_inc_05 = -17.3e6;
    y_plot_max_delta_1 =  100; % covers both slip patches (in detail), m
    y_plot_min_delta_1 = -100; % covers both slip patches (in detail), m
    y_plot_max_delta_2 =   80; % covers top patch only (further detail), m
    y_plot_min_delta_2 =   60; % covers top patch only (further detail), m
    y_plot_max_sigma   =  250; % covers both slip patches (overview), m
    y_plot_min_sigma   = -250; % covers both slip patches (overview), m
    y_plot_n_pt = 2000; % number of points between y_plot_min_... and y_plot_max_...
    % (only relevant for some plots)
    y_plot_max_bounds =  100; % covers both slip patches (in detail), m
    y_plot_min_bounds = -100; % covers both slip patches (in detail), m
    blue = [0/255 128/255 255/255];
    gray = [235/255 235/255 235/255];
    green = [0/255 204/255 0/255];
    orange = [255/255 185/255 0/255];
    colors = [blue; gray; green; orange];
    fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
        D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
        eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
        flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
        flag_plot_eig,flag_plot_snap,flag_reg,flag_top_only,...
        g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
        p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
        p_max_bounds,p_min_bounds,...
        phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
        y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
        y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
        y_plot_n_pt);
    
    % Regularisation plot:
    fault_slip_plot_regularizations
end
